const logBox = document.getElementById('logOverlay');

function ts() {
  return (
    new Date().toLocaleTimeString('cs-CZ', { hour12: false }) +
    '.' +
    String(new Date().getMilliseconds()).padStart(3, '0')
  );
}

function log(msg, ...rest) {
  const line = `[${ts()}] ${msg} ${rest.map(String).join(' ')}`;
  console.log(line);

  if (logBox) {
    logBox.textContent += line + '\n';
    if (logBox.textContent.length > 2000) {
      logBox.textContent = logBox.textContent.slice(-2000);
    }
    logBox.scrollTop = logBox.scrollHeight;
  }
}

const video = document.getElementById('video');
const canvas = document.getElementById('overlay');
const shotBtn = document.getElementById('shot');
const emojiDiv = document.getElementById('emoji');

const API_URL = 'https://emo-api.infis-projekty.eu/upload';
const HISTORY_MS = 500;

//0 emoji, 1 infisák
const USE_IMAGE_EMOJI = 1;

const EMOJI = {
  neutral: '😐',
  happy: '😄',
  sad: '😢',
  angry: '😡',
  fearful: '😱',
  disgusted: '🤢',
  surprised: '😲'
};

let expressionHistory = [];


function computeDominantEmotion() {

  if (expressionHistory.length) {
    const avg = {};
    expressionHistory.forEach(({ expressions }) => {
      for (const [k, v] of Object.entries(expressions)) {
        avg[k] = (avg[k] || 0) + v;
      }
    });
    for (const key in avg) {
      avg[key] /= expressionHistory.length;
    }

    return Object.entries(avg).sort((a, b) => b[1] - a[1])[0][0];
  }

  return video.dataset.emotion || 'neutral';
}

function updateEmojiWithUnicode() {
  const dominant = computeDominantEmotion();
  if (emojiDiv.dataset.currentEmotion !== dominant) {
    log('[EMOJI] (text) →', dominant);
    emojiDiv.textContent = EMOJI[dominant] || EMOJI['neutral'];
    emojiDiv.dataset.currentEmotion = dominant;
    video.dataset.emotion = dominant;
    emojiDiv.classList.add('pulse');
    setTimeout(() => emojiDiv.classList.remove('pulse'), 800);
  }
}

function updateEmojiWithImage() {
  const dominant = computeDominantEmotion();
  if (emojiDiv.dataset.currentEmotion !== dominant) {
    log('[EMOJI] (img) →', dominant);
    emojiDiv.innerHTML = `<img src="emotes/${dominant}.png" alt="${dominant}" />`;
    emojiDiv.dataset.currentEmotion = dominant;
    video.dataset.emotion = dominant;
    emojiDiv.classList.add('pulse');
    setTimeout(() => emojiDiv.classList.remove('pulse'), 800);
  }
}

let updateEmojiDisplay;
if (USE_IMAGE_EMOJI) {
  updateEmojiDisplay = updateEmojiWithImage;
} else {
  updateEmojiDisplay = updateEmojiWithUnicode;
}

(async () => {
  log('[BOOT] start – backend WASM');
  tf.wasm.setWasmPaths(
    'https://cdn.jsdelivr.net/npm/@tensorflow/tfjs-backend-wasm@4.22.0/dist/'
  );
  await tf.setBackend('wasm');
  await tf.ready();
  log('[BOOT] TF backend =', tf.getBackend());

  console.time('[BOOT] load models');
  await faceapi.nets.tinyFaceDetector.loadFromUri(
    'models/tiny_face_detector_model-weights_manifest.json'
  );
  await faceapi.nets.faceExpressionNet.loadFromUri(
    'models/face_expression_model-weights_manifest.json'
  );
  console.timeEnd('[BOOT] load models');
  log('[BOOT] models ready');

  canvas.width = 1920;
  canvas.height = 1080;

  startCamera();
})();

function startCamera() {
  console.time('[CAMERA] getUserMedia');
  navigator.mediaDevices
    .getUserMedia({ video: { width: 1920, height: 1080 } })
    .then((stream) => {
      console.timeEnd('[CAMERA] getUserMedia');
      video.srcObject = stream;
      video.onloadedmetadata = () => {
        video.play();
        detectLoop();
      };
    })
    .catch((e) => {
      log('Nelze spustit kameru:', e.message);
      alert('Kamera se nespustila – zkontroluj oprávnění');
    });
}

async function detectLoop() {
  if (video.readyState === 4) {
    const face = await faceapi
      .detectSingleFace(
        video,
        new faceapi.TinyFaceDetectorOptions({ scoreThreshold: 0.5 })
      )
      .withFaceExpressions();

    const now = Date.now();
    if (face) {
      expressionHistory.push({ time: now, expressions: face.expressions });
      expressionHistory = expressionHistory.filter(
        (e) => now - e.time <= HISTORY_MS
      );
    }

    updateEmojiDisplay();
  }
  requestAnimationFrame(detectLoop);
}

async function takePhoto() {
  console.time('[SHOT] total');
  log('[SHOT] click');

  const snap = document.createElement('canvas');
  snap.width = 1920;
  snap.height = 1080;

  const sc = snap.getContext('2d');
  sc.drawImage(video, 0, 0, snap.width, snap.height);
  sc.drawImage(canvas, 0, 0, snap.width, snap.height);

  console.time('[SHOT] toBlob');
  const blob = await new Promise((r) => snap.toBlob(r, 'image/jpeg', 0.9));
  console.timeEnd('[SHOT] toBlob');

  const emotion = video.dataset.emotion || 'neutral';
  log('[SHOT] emotion =', emotion);

  const form = new FormData();
  form.append('emotion', emotion);
  form.append('face', blob, 'face.jpg');

  try {
    console.time('[SHOT] fetch');
    const resp = await fetch(API_URL, { method: 'POST', body: form });
    const data = await resp.json();
    console.timeEnd('[SHOT] fetch');
    if (!data.success) throw new Error('remote error');
    log('[SHOT] uploaded id =', data.id);
  } catch (err) {
    log('[SHOT] FAIL', err.message);
    alert('Upload selhal');
  }
  console.timeEnd('[SHOT] total');
}

function startCountdown() {
  let count = 5;
  shotBtn.classList.add('counting');
  const countdownSpan = shotBtn.querySelector('.countdown');
  countdownSpan.textContent = count;

  const interval = setInterval(() => {
    count--;
    if (count >= 0) {
      countdownSpan.textContent = count;
    }
    if (count < 0) {
      clearInterval(interval);
      shotBtn.classList.remove('counting');
      countdownSpan.textContent = '';
      takePhoto();
    }
  }, 1000);
}

shotBtn.addEventListener('click', () => {
  if (!shotBtn.classList.contains('counting')) {
    startCountdown();
  }
});

window.addEventListener('keydown', (e) => {
  if (e.code === 'Space' && !shotBtn.classList.contains('counting')) {
    e.preventDefault();
    startCountdown();
  }
});
