const logBox = document.getElementById('logOutput');
const modelsList = document.getElementById('models-list');
const resultsJson = document.getElementById('results-json');
const video = document.getElementById('video');
const canvas = document.getElementById('overlay');
const ctx = canvas.getContext('2d');

function ts() {
  return new Date().toLocaleTimeString('cs-CZ', { hour12: false }) +
    '.' + String(new Date().getMilliseconds()).padStart(3, '0');
}

function log(message, ...rest) {
  const line = `[${ts()}] ${message} ${rest.join(' ')}\n`;
  logBox.textContent += line;
  if (logBox.scrollHeight > logBox.clientHeight) {
    logBox.scrollTop = logBox.scrollHeight;
  }
}

(async () => {
  log('Inicializace – nastavení WASM backendu');
  tf.wasm.setWasmPaths(
    'https://cdn.jsdelivr.net/npm/@tensorflow/tfjs-backend-wasm@4.22.0/dist/'
  );
  await tf.setBackend('wasm');
  await tf.ready();
  log('TF backend =', tf.getBackend());

  const modelInfos = [
    { name: 'SSD Mobilenet v1', load: faceapi.nets.ssdMobilenetv1.loadFromUri('models/ssd_mobilenetv1_model-weights_manifest.json') },
    { name: 'Tiny Face Detector', load: faceapi.nets.tinyFaceDetector.loadFromUri('models/tiny_face_detector_model-weights_manifest.json') },
    { name: 'Face Landmark 68', load: faceapi.nets.faceLandmark68Net.loadFromUri('models/face_landmark_68_model-weights_manifest.json') },
    { name: 'Face Landmark 68 Tiny', load: faceapi.nets.faceLandmark68TinyNet.loadFromUri('models/face_landmark_68_tiny_model-weights_manifest.json') },
    { name: 'Face Expression', load: faceapi.nets.faceExpressionNet.loadFromUri('models/face_expression_model-weights_manifest.json') },
    { name: 'Age & Gender', load: faceapi.nets.ageGenderNet.loadFromUri('models/age_gender_model-weights_manifest.json') },
    { name: 'Face Recognition', load: faceapi.nets.faceRecognitionNet.loadFromUri('models/face_recognition_model-weights_manifest.json') }
  ];

  log('Načítání modelů:');
  for (const info of modelInfos) {
    const li = document.createElement('li');
    li.textContent = info.name;
    modelsList.appendChild(li);
  }

  console.time('LOAD ALL MODELS');
  await Promise.all(modelInfos.map(info => info.load));
  console.timeEnd('LOAD ALL MODELS');
  log('Všechny modely načteny');

  startCamera();
})();

function startCamera() {
  log('Spuštění kamery: požadavek na 640×480');
  navigator.mediaDevices.getUserMedia({ video: { width: 640, height: 480 } })
    .then(stream => {
      video.srcObject = stream;
      video.onloadedmetadata = () => {
        video.play();
        detectLoop();
        log('Kamera spuštěna');
      };
    })
    .catch(err => {
      log('Chyba při přístupu ke kameře:', err.message);
    });
}

async function detectLoop() {
  if (video.readyState === 4) {
    const detections = await faceapi.detectAllFaces(video, new faceapi.SsdMobilenetv1Options({ minConfidence: 0.5 }))
      .withFaceLandmarks()
      .withFaceExpressions()
      .withAgeAndGender()
      .withFaceDescriptors();

    ctx.clearRect(0, 0, canvas.width, canvas.height);

    const displayData = [];
    for (const det of detections) {
      const { box } = det.detection;
      ctx.strokeStyle = '#4a8ff0';
      ctx.lineWidth = 2;
      ctx.strokeRect(box.x, box.y, box.width, box.height);

      const landmarks = det.landmarks;
      ctx.fillStyle = '#4a8ff0';
      for (const pt of landmarks.positions) {
        ctx.fillRect(pt.x - 1, pt.y - 1, 2, 2);
      }

      const expressions = det.expressions;
      const topExpr = Object.entries(expressions).sort((a, b) => b[1] - a[1])[0];
      ctx.fillStyle = '#fff';
      ctx.fillRect(box.x, box.y - 24, box.width, 24);
      ctx.fillStyle = '#1524a9';
      ctx.font = '16px sans-serif';
      ctx.fillText(`${topExpr[0]} (${topExpr[1].toFixed(2)})`, box.x + 4, box.y - 6);

      const age = det.age.toFixed(0);
      const gender = det.gender;
      ctx.fillText(`${gender}, ${age} let`, box.x + 4, box.y + box.height + 18);

      displayData.push({
        box: { x: box.x, y: box.y, width: box.width, height: box.height },
        age: det.age,
        gender: det.gender,
        expressions: det.expressions,
        descriptor: Array.from(det.descriptor)
      });
    }

    resultsJson.textContent = JSON.stringify(displayData, null, 2);
  }
  requestAnimationFrame(detectLoop);
}
